<?php

class ReservationFormStatus {
    const BEGIN_FORM_ENTRY = 1;
    const CONFIRM_FORM = 2;
    const EMAIL_FORM = 3;
}

/**
 * PageWorker
 *
 * @author Alexander Bandisch
 * @copyright Alexander Bandisch
 * @version 2014
 * @access public
 */
class PageWorker {

    protected $_errorMessage;
    protected $_successMessage;
    protected $_postData;
    protected $_currentPage;
    protected $_libPath;
    protected $_failedValidationFields;

    protected $_reservation;


    /**
     * PageWorker::__construct()
     * @param array $menuItems
     */
    public function __construct() {
        /* Set the default error and success messages */
        $this->setErrorMessage("");
        $this->setsuccessMessage("");
        $this->setLibPath('library/');

    }

    public function processPage() {
        if ($this->isPost()) {
            $this->setPostData($_POST);
            if ($this->isSetPostField('Continue')) { // Continue button was pressed, validate the form
                if ($this->validateSubmittedDetails($this->getPostData())) {
                    $this->setCurrentPage(ReservationFormStatus::CONFIRM_FORM);
                } else {
                    $this->setCurrentPage(ReservationFormStatus::BEGIN_FORM_ENTRY);
                }
            }

            if ($this->isSetPostField('Confirm')) { // Continue button was pressed, validate the form

                $this->_reservation = new Reservation();
                $this->_reservation->setDetails($this->getPostData());
                $this->sendGuestEmail();
                $this->sendHotelEmail();
                $this->setCurrentPage(ReservationFormStatus::EMAIL_FORM);
            }
        } else {
            $this->setCurrentPage(ReservationFormStatus::BEGIN_FORM_ENTRY);
        }
    }

    public function currentPageContents() {
        switch ($this->getCurrentPage()) {
            case ReservationFormStatus::BEGIN_FORM_ENTRY:
                return $this->getLibPath() . 'reservationsForm.php';
                break;
            case ReservationFormStatus::CONFIRM_FORM:
                return $this->getLibPath() . 'reservationsFormConfirm.php';
                break;
            case ReservationFormStatus::EMAIL_FORM:
                return $this->getLibPath() . 'reservationsFormComplete.php';
                break;
            default:
                return $this->getLibPath() . 'reservationsForm.php';
        }
    }

    public function getReservationProperty($property) {
        return $this->_reservation->getProperty($property);
    }

    protected function sendGuestEmail() {
        $Emailer = new Emailer();

        $EmailHTML = $this->createEmailBody($this->getLibPath() . "../emails/guest-email-html.php", $this->_reservation->toArray());
        $EmailTXT = $this->createEmailBody($this->getLibPath() . "../emails/guest-email-txt.php", $this->_reservation->toArray());

        $Emailer->sendEmail('Rabaul Hotel - Online Booking', $EmailHTML, $EmailTXT, $this->_reservation->getFirstName() . ' ' . $this->_reservation->getLastName(), $this->_reservation->getEmailAddress());
    }

    protected function sendHotelEmail() {
        $Emailer = new Emailer();

        $EmailHTML = $this->createEmailBody($this->getLibPath() . "../emails/hotel-email-html.php", $this->_reservation->toArray());
        $EmailTXT = $this->createEmailBody($this->getLibPath() . "../emails/hotel-email-txt.php", $this->_reservation->toArray());

        $Emailer->sendEmail('Rabaul Hotel - Online Booking', $EmailHTML, $EmailTXT, MailDetails::HOTEL_EMAIL_NAME, MailDetails::HOTEL_EMAIL);

    }

    /**
     * createEmailBody - takes a template and array of details and substitues in the details into the template and returns the updated template
     * @param string - path to the email template
     * @param array - array with the details to substitute into the template. Array is of the format "%FIND%" => "REPLACE"
     * @return string - updated email template
     */
    protected function createEmailBody($pathToTemplate, $details = array()) {
        $email_body = file_get_contents($pathToTemplate);

        foreach ($details as $find => $replace) {
            $email_body = str_replace('%' . $find . '%', $replace, $email_body);
        }
        return $email_body;
    }

    protected function validateSubmittedDetails($formData) {
        $validator = new gump();

        /* Validation Rules/Filters */
        $rules = array(
            'FirstName'     => 'required',
            'LastName'      => 'required',
            'EmailAddress'  => 'required',
            'Mobile'        => 'required',
            'StreetName'    => 'required',
            'City'          => 'required',
            'Postcode'      => 'required',
            'StateProvince' => 'required',
            'CheckInDate'   => 'required',
            'CheckOutDate'  => 'required'
        );

        $filters = array(
            'FirstName'     => 'trim',
            'LastName'      => 'trim',
            'EmailAddress'  => 'trim',
            'Mobile'        => 'trim',
            'StreetName'    => 'trim',
            'City'          => 'trim',
            'Postcode'      => 'trim',
            'StateProvince' => 'trim',
            'CheckInDate'   => 'trim',
            'CheckOutDate'  => 'trim'
        );
        $error_messages = array(
            'FirstName'     => 'First name is required',
            'LastName'      => 'Last name is required',
            'EmailAddress'  => 'Email address is required',
            'Mobile'        => 'Mobile number is required',
            'StreetName'    => 'Street name and number is required',
            'City'          => 'City is required',
            'Postcode'      => 'Postcode is required',
            'StateProvince' => 'State or Province is required',
            'CheckInDate'   => 'Check-in date is required',
            'CheckOutDate'  => 'Check-out date is required'
        );
        $this->setPostData($validator->filter($formData, $filters));
        $validated = $validator->validate($formData, $rules);
        $FailedValidationFields = array();

        if ($validated !== true) {
            foreach ($validated as $number => $failedValidatedData) {
                $FailedValidationFields[$failedValidatedData["field"]] = $error_messages[$failedValidatedData["field"]];
            }
        }

        $this->_failedValidationFields = $FailedValidationFields;

        return (count($this->_failedValidationFields) == 0 ? true : false);

    }


    /**
     * PageWorker::fieldFailedValidation()
     * Checks given field has failed validation.
     *
     * @param string - input name to check for validation
     * @return bool - true if validation failed, false if validation successful
     */
    public function fieldFailedValidation($field) {
        return isset($this->_failedValidationFields[$field]);
    }

    /**
     * PageWorker::failedValidationMessage
     * Returns the given validation message.
     *
     * @param string - input name to get corresponding validation message
     * @return string - validation message
     */
    public function failedValidationMessage($field) {
        $message = '<div class="col-sm-offset-4 col-sm-8"><label class="has-error control-label" for="' . $field . '">%ERROR_MESSAGE%</label></div>';

        $message = str_replace("%ERROR_MESSAGE%", $this->_failedValidationFields[$field], $message);

        return $message;
    }

    /**
     * PageWorker::isSetPost()
     * Determines if the field is set in POST
     *
     * @return bool - true if set, false if not
     */
    public function isSetPostField($field) {
        return isset($this->_postData[$field]);
    }

    /**
     * PageWorker::isPost()
     * Determines if the REQUEST METHOD is POST or not
     *
     * @return bool - true if POST, false if not
     */
    public function isPost() {
        return ($_SERVER['REQUEST_METHOD'] == 'POST');
    }

    /**
     * PageWorker::getFieldPOSTValue - return the given field value after a POST submit, if any
     * @param string - input name
     * @return string - POST submit value
     */
    public function getFieldPOSTValue($field) {
        return isset($this->_postData[$field]) ? $this->_postData[$field] : '';
    }

    /**
     * PageWorker::POSTFieldExists - returns true if the given field POST field exists
     * @param string - input name
     * @return bool - returns true if the given field POST field exists
     */
    public function POSTFieldExists($field) {
        return isset($this->_postData[$field]);
    }

    /**
     * PageWorker::errorMessageExists()
     * Determines if an error message has been set
     *
     * @return bool - true if error message set, false if not
     */
    public function errorMessageExists() {
        return ($this->getErrorMessage() != "");
    }

    /**
     * PageWorker::successMessageExists()
     * Determines if a success message has been set
     *
     * @return bool - true if success message set, false if not
     */
    public function successMessageExists() {
        return ($this->getsuccessMessage() != "");
    }

    /**
     * PageWorker::getPostData()
     * Returns the $_POST data
     *
     * @return array - POST data array
     */
    public function getPostData() {
        return $this->_postData;
    }

    /**
     * PageWorker::setPostData()
     * Sets the POST data to the values of $_POST
     *
     * @param array $value - POST data
     */
    public function setPostData($value) {
        $this->_postData = $value;
    }

    /**
     * PageWorker::getLibPath()
     * method for returning the library path
     *
     * @return string - path to Library
     */
    protected function getLibPath() {
        return $this->_libPath;
    }

    /**
     * PageWorker::setLibPath()
     * Sets the library path
     *
     * @param string - value to set
     */
    protected function setLibPath($value) {
        $this->_libPath = str_replace('classes', $value, dirname(__FILE__));
    }

    /**
     * PageWorker::getErrorMessage()
     * Returns the error message
     *
     * @return string - error message
     */
    public function getErrorMessage() {
        return $this->_errorMessage;
    }

    /**
     * PageWorker::setErrorMessage()
     * Sets the error message
     *
     * @param string $value - error message
     */
    public function setErrorMessage($value) {
        $this->_errorMessage = $value;
    }

    /**
     * PageWorker::getSuccessMessage()
     * Gets the success message
     *
     * @return string - success message
     */
    public function getSuccessMessage() {
        return $this->_successMessage;
    }

    /**
     * PageWorker::setSuccessMessage()
     * Sets the success message
     *
     * @param string $value - success message
     */
    public function setSuccessMessage($value) {
        $this->_successMessage = $value;
    }

    /**
     * PageWorker::getCurrentPage()
     * Gets the current page
     *
     * @return string - current page
     */
    public function getCurrentPage() {
        return $this->_currentPage;
    }

    /**
     * PageWorker::setCurrentPage()
     * Sets the current page
     *
     * @param string $value - current page
     */
    public function setCurrentPage($value) {
        $this->_currentPage = $value;
    }

}

?>